<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\DB;

class Patient extends Model
{
    use HasFactory;

    protected $fillable = [
        'name',
        'species',
        'breed',
        'age',
        'age_months',
        'owner_id',
        'notes',
        'passbook_number'
    ];

    protected static function booted()
    {
        // Ensure a passbook number exists before insert so DB constraints
        // (non-null, unique) do not fail. We attempt to use the next
        // AUTO_INCREMENT value for a sequential number; if that fails
        // fall back to a timestamp-based value.
        static::creating(function ($patient) {
            if (empty($patient->passbook_number)) {
                $patient->passbook_number = static::previewNextPassbookNumber();
            }
        });

        // After creation, prefer the definitive `PB-{id}` format using the
        // real id assigned by the database. Save quietly to avoid event loops.
        static::created(function ($patient) {
            $expected = static::generatePassbookNumber($patient->id);
            if ($patient->passbook_number !== $expected) {
                $patient->passbook_number = $expected;
                $patient->saveQuietly();
            }
        });
    }

    /**
     * Generate a passbook number. If an $id is provided, use it so the
     * passbook number is sequential and deterministic; otherwise fall back
     * to the previous random strategy.
     */
    public static function generatePassbookNumber($id = null)
    {
        // Use simple sequential format PB-{id} when we have an id.
        if ($id !== null) {
            return 'PB-' . (string) $id;
        }

        // fallback: generate a PB-{timestamp} if no id available
        return 'PB-' . time();
    }

    /**
     * Preview the next passbook number using the next AUTO_INCREMENT value
     * for the `patients` table. Falls back to generatePassbookNumber()
     * if AUTO_INCREMENT cannot be determined.
     */
    public static function previewNextPassbookNumber()
    {
        try {
            $row = DB::selectOne("SELECT `AUTO_INCREMENT` as next FROM INFORMATION_SCHEMA.TABLES WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = 'patients'");
            if ($row && isset($row->next)) {
                return static::generatePassbookNumber((int) $row->next);
            }
        } catch (\Throwable $e) {
            // ignore and fall back
        }
        return static::generatePassbookNumber();
    }

    public function owner()
    {
        return $this->belongsTo(Owner::class);
    }

    public function vaccinations()
    {
        return $this->hasMany(PatientVaccination::class);
    }
}
